<?php

namespace App\Http\Controllers\Component2;

use App\Http\Controllers\Controller;
use App\Models\C2WeofBeneficiary;
use App\Models\C1Group;
use Illuminate\Http\Request;
use App\Exports\C2WeofBeneficiariesExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\C2SeedlabBeneficiary;
use App\Models\C2GrowthlabBusiness;
use App\Models\C2Grant;
use Illuminate\Support\Facades\DB;

class C2WeofBeneficiaryController extends Controller
{
    public function index(Request $request)
    {
        $query = C2WeofBeneficiary::with('group');

        // Optional: Add filters later (county, payam, accelerator, etc.)
        if ($request->filled('county')) {
            $query->where('county', $request->county);
        }
        if ($request->filled('accelerator')) {
            $query->where('accelerator_participation', true);
        }

        $query = C2WeofBeneficiary::with('group');

        $beneficiaries = $query->latest()->paginate(15)->withQueryString();

        $stats = [
            'total'              => C2WeofBeneficiary::count(),
            'marital_status'         => C2WeofBeneficiary::where('marital_status', 'married')->count(), 
            'accelerator'        => C2WeofBeneficiary::where('accelerator_participation', true)->count(),
            'graduated'          => C2WeofBeneficiary::where('accelerator_graduation', true)->count(),
            'grant_received'     => C2WeofBeneficiary::whereNotNull('grant_1st_tranche')
                                        ->orWhereNotNull('grant_2nd_tranche')->count(),
            'business_registered'=> C2WeofBeneficiary::where('registration_status', true)->count(),
        ];

        $countyCounts = C2WeofBeneficiary::select('county', DB::raw('count(*) as total'))
            ->whereNotNull('county')
            ->groupBy('county')
            ->pluck('total', 'county')
            ->toArray();

        $countyCoords = [
            'Juba' => ['lat' => 4.8594, 'lng' => 31.5713],
            'Wau' => ['lat' => 7.7011, 'lng' => 27.9953],
            'Malakal' => ['lat' => 9.5334, 'lng' => 31.6605],
            'Yambio' => ['lat' => 4.5721, 'lng' => 28.3955],
            'Torit' => ['lat' => 4.4133, 'lng' => 32.5678],
            'Bor' => ['lat' => 6.2092, 'lng' => 31.5589],
            'Rumbek' => ['lat' => 6.8116, 'lng' => 29.6912],
            'Aweil' => ['lat' => 8.7680, 'lng' => 27.3966],
            'Yei' => ['lat' => 4.0951, 'lng' => 30.6773],
            'Bentiu' => ['lat' => 9.2573, 'lng' => 29.8037],
            'Kuajok' => ['lat' => 8.3090, 'lng' => 27.9960],
            // Add other counties as needed based on your DB data
        ];

        // 3. Prepare data for the frontend
        $mapData = [];
        foreach ($countyCounts as $countyName => $count) {
            // Fuzzy matching or direct check
            $coords = $countyCoords[$countyName] ?? null; 
            
            // Default to center if unknown, or skip. Here we skip if no coords found.
            if ($coords) {
                $mapData[] = [
                    'name' => $countyName,
                    'lat' => $coords['lat'],
                    'lng' => $coords['lng'],
                    'count' => $count
                ];
            }
        }

        return view('component2.index', compact('beneficiaries', 'stats', 'mapData'));
    }

    public function create()
    {
        $groups = C1Group::orderBy('name')->pluck('name', 'id')->prepend('— No Group —', '');
        return view('component2.create', compact('groups'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'group_id'              => 'nullable|exists:c1_groups,id',
            'woef_id'               => 'required|string|unique:c2_weof_beneficiaries,woef_id|max:100',
            'name'                  => 'required|string|max:191',
            'age'                   => 'nullable|integer|min:15|max:100',
            'marital_status'        => 'nullable|in:married,not_married,divorced,widowed',
            'resident_status'       => 'nullable|string',
            'county'                => 'required|string',
            'payam'                 => 'required|string',
            'education'             => 'nullable|string',
            'phone_no'              => 'nullable|string|max:20',
            'business_name'         => 'required|string',
            'business_category'     => 'nullable|string',
            'business_type'         => 'nullable|string',
            'establishment_year'    => 'nullable|integer|min:1900|max:' . date('Y'),
            'registration_status'   => 'boolean',
            'monthly_revenue'       => 'nullable|numeric',
            'ever_received_grant'   => 'boolean',
            'ever_received_loan'    => 'boolean',
            'accelerator_participation' => 'boolean',
            'accelerator_graduation'    => 'boolean',
            'grant_1st_tranche'     => 'nullable|numeric',
            'grant_2nd_tranche'     => 'nullable|numeric',
        ]);

        C2WeofBeneficiary::create($validated);

        return redirect()->route('component2.index')->with('success', 'WEOF Beneficiary added successfully!');
    }

    public function edit(C2WeofBeneficiary $beneficiary)
    {
        $groups = C1Group::orderBy('name')->pluck('name', 'id')->prepend('— No Group —', '');
        return view('component2.edit', compact('beneficiary', 'groups'));
    }

    public function update(Request $request, C2WeofBeneficiary $beneficiary)
    {
        $validated = $request->validate([
            'group_id'              => 'nullable|exists:c1_groups,id',
            'woef_id'               => 'required|string|max:100|unique:c2_weof_beneficiaries,woef_id,' . $beneficiary->id,
            'name'                  => 'required|string|max:191',
            'age'                   => 'nullable|integer|min:15|max:100',
            'marital_status'        => 'nullable|in:married,not_married,divorced,widowed',
            'resident_status'       => 'nullable|string',
            'county'                => 'required|string',
            'payam'                 => 'required|string',
            'education'             => 'nullable|string',
            'phone_no'              => 'nullable|string|max:20',
            'business_name'         => 'required|string',
            'business_category'     => 'nullable|string',
            'business_type'         => 'nullable|string',
            'establishment_year'    => 'nullable|integer|min:1900|max:' . date('Y'),
            'registration_status'   => 'boolean',
            'monthly_revenue'       => 'nullable|numeric',
            'ever_received_grant'   => 'boolean',
            'ever_received_loan'    => 'boolean',
            'accelerator_participation' => 'boolean',
            'accelerator_graduation'    => 'boolean',
            'grant_1st_tranche'     => 'nullable|numeric',
            'grant_2nd_tranche'     => 'nullable|numeric',
        ]);

        $beneficiary->update($validated);

        return redirect()->route('component2.index')->with('success', 'Beneficiary updated successfully!');
    }

    public function destroy(C2WeofBeneficiary $beneficiary)
    {
        $beneficiary->delete();
        return back()->with('success', 'Beneficiary deleted.');
    }

    public function export()
    {
        return Excel::download(
            new C2WeofBeneficiariesExport,
            'WEOF_Beneficiaries_' . now()->format('Y-m-d') . '.xlsx'
        );
    }

    public function dashboard()
    {
        // Combined beneficiaries
        $seedlab = C2SeedlabBeneficiary::with('grant')->get();
        $growthlab = C2GrowthlabBusiness::with('grant')->get();
        $all = $seedlab->concat($growthlab);

        // Stats
        $stats = [
            'total_beneficiaries' => $all->count(),
            'female_count' => $all->whereIn('gender', ['Female','Adol.Girl'])
                                ->whereNotNull('gender')->count(),
            'total_disbursed' => C2Grant::whereIn('grantable_type', [
                C2SeedlabBeneficiary::class,
                C2GrowthlabBusiness::class
            ])->sum('total_received'),
            'utilization_rate' => C2Grant::where('utilized_for_plan', 'Yes')->count() /
                                (C2Grant::count() ?: 1) * 100,
        ];

        // Chart Data
        $chartData = [
            'sectors' => [
                'labels' => $all->groupBy('business_sector')->keys()->toArray(),
                'data' => $all->groupBy('business_sector')->map->count()->values()->toArray()
            ],
            'counties' => [
                'labels' => $all->whereNotNull('county')->groupBy('county')->keys()->take(10)->toArray(),
                'data' => $all->whereNotNull('county')->groupBy('county')->map->count()->values()->take(10)->toArray()
            ],
            'grants' => [
                'disbursed' => C2Grant::where('total_received', '>', 0)->count(),
                'approved_not_paid' => C2Grant::where('total_approved', '>', 0)
                                            ->where('total_received', 0)->count(),
                'no_grant' => ($seedlab->count() + $growthlab->count()) - C2Grant::count()
            ],
            'cohorts' => [
                'labels' => $seedlab->pluck('cohort_year')->unique()->sort()->values(),
                'total' => $seedlab->groupBy('cohort_year')->map->count()->values(),
                'completed' => $seedlab->where('is_fully_trained', true)
                                    ->groupBy('cohort_year')->map->count()->values()
            ],
            'modules' => collect(range(1,8))->map(function($i) use ($seedlab) {
                return $seedlab->where("module_$i", 'Completed')->count() / ($seedlab->count() ?: 1) * 100;
            })->toArray()
        ];

        return view('component2.dashboard', compact('stats', 'chartData'));
    }
}