<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class C1Group extends Model
{
    use SoftDeletes;

    protected $table = 'c1_groups';

    protected $fillable = [
        'name', 'nature', 'state', 'county', 'payam',
        'establishing_org_name', 'establishment_type', 'support_received_details',
        'date_formation', 'vsla_registration_date', 'main_activities',
        'total_members', 'num_women', 'num_adolescent_girls', 'num_men', 'num_adolescent_boys',
        'num_women_disability', 'num_widows', 'num_women_heads_hh', 'leadership',
        'num_saving_cycles_completed', 'total_credit_given_out',
        'num_members_investing_existing', 'num_members_started_new',
        'has_livelihood_plan', 'livelihood_plan_validated',
        'received_in_kind_support_plan', 'total_value_in_kind',
        'num_members_received_in_kind', 'num_female_received_in_kind', 'num_male_received_in_kind',
        'initiated_regular_savings', 'created_new_business', 'type_new_business',
        'initiated_credit', 'num_benefited_credit', 'num_dropouts', 'month'
    ];

    protected $casts = [
        'date_formation' => 'date',
        'vsla_registration_date' => 'date',
        'month' => 'date',
        'nature' => 'string',
        'establishment_type' => 'string',
        'leadership' => 'string',
        'has_livelihood_plan' => 'boolean',
        'livelihood_plan_validated' => 'boolean',
        'received_in_kind_support_plan' => 'boolean',
        'initiated_regular_savings' => 'boolean',
        'created_new_business' => 'boolean',
        'initiated_credit' => 'boolean',
    ];

    // === Relationships ===

    /**
     * Members of this VSLA/Community Group
     */
    public function members(): HasMany
    {
        return $this->hasMany(C1Member::class, 'group_id');
    }

    /**
     * Supports (in-kind, equipment, grants) given to the group
     */
    public function supports(): MorphMany
    {
        return $this->morphMany(C1Support::class, 'supportable');
    }

    /**
     * WEOF beneficiaries optionally linked to this group
     */
    public function weofBeneficiaries(): HasMany
    {
        return $this->hasMany(C1WeofBeneficiary::class, 'group_id');
    }

    // === Scopes ===

    public function scopeActive($query)
    {
        return $query->whereHas('members', fn($q) => $q->where('active', true));
    }

    public function scopeByCounty($query, $county)
    {
        return $query->where('county', $county);
    }

    public function scopeByPayam($query, $payam)
    {
        return $query->where('payam', $payam);
    }

    // === Accessors ===

    public function getTotalFemaleMembersAttribute(): int
    {
        return $this->num_women + $this->num_adolescent_girls;
    }

    public function getTotalMaleMembersAttribute(): int
    {
        return $this->num_men + $this->num_adolescent_boys;
    }

    public function getTotalMembersCalculatedAttribute(): int
    {
        return $this->totalFemaleMembers + $this->totalMaleMembers;
    }

    public function getGbvVulnerableCountAttribute(): int
    {
        return $this->num_women_disability + $this->num_widows + $this->num_women_heads_hh;
    }
}