<?php

namespace App\Http\Controllers\Component1;

use App\Http\Controllers\Controller;
use App\Models\C1Group;
use App\Models\C1Member;
use App\Models\C1Training;
use App\Exports\VslaMonthlyReportExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;

class VslaReportController extends Controller
{
    public function index()
    {
        // 1. VSLA Engagement Data (Mature/Progressive/New)
        $engagement = C1Group::selectRaw('
                nature as category,
                COUNT(*) as achieved
            ')
            ->groupBy('nature')
            ->get()
            ->keyBy('category');

        $targets = [
            'mature' => 240,
            'progressive' => 160,
            'new' => 400,
            'total' => 800
        ];

        $engagementData = collect(['mature', 'progressive', 'new', 'total'])->map(function ($cat) use ($engagement, $targets) {
            $achieved = $engagement->get($cat)->achieved ?? 0;
            $target = $targets[$cat] ?? ($cat === 'total' ? 800 : 0);
            $progress = $target > 0 ? round(($achieved / $target) * 100, 2) : 0;

            return [
                'category' => ucfirst($cat),
                'target' => $target,
                'achieved' => $achieved,
                'progress' => $progress . '%'
            ];
        });

        // 2. Membership Breakdown
        $membership = C1Group::selectRaw("
                nature,
                SUM(num_women) as female,
                SUM(num_men) as male,
                SUM(num_adolescent_girls) as adol_girls,
                SUM(total_members) as total
            ")
            ->groupBy('nature')
            ->get()
            ->keyBy('nature');

        // 3. Training & Support Aggregates
        $trainedMembers = C1Member::where('completed_core_1', true)
            ->orWhere('completed_core_2', true)
            ->orWhere('completed_core_3', true)
            ->orWhere('completed_core_4', true)
            ->count();

        $kitsProvided = C1Group::where('received_certificate', true)->count();
        $productiveAssets = C1Group::where('received_in_kind_support_plan', true)->sum('total_value_in_kind');

        $newBusinesses = C1Group::where('created_new_business', true)->count();

        // 4. Chart Data for Dashboard
        $chartLabels = $engagementData->pluck('category')->toArray();
        $chartAchieved = $engagementData->pluck('achieved')->toArray();
        $chartTargets = $engagementData->pluck('target')->toArray();

        return view('component1.reports.vsla_summary', compact(
            'engagementData',
            'membership',
            'trainedMembers',
            'kitsProvided',
            'productiveAssets',
            'newBusinesses',
            'chartLabels',
            'chartAchieved',
            'chartTargets'
        ));
    }
    public function export()
    {
        return Excel::download(new VslaMonthlyReportExport, 'VSLA_Monthly_Report_' . now()->format('Y_m_d') . '.xlsx');
    }
}