<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class C1Indicator extends Model
{
    use SoftDeletes;

    /** @var string */
    protected $table = 'c1_indicators';

    /** @var bool */
    public $timestamps = true; // created_at & updated_at

    /** @var array */
    protected $fillable = [
        'kpi_name',
        'sl_target',
        'sl_definition',
        'gl_target',
        'gl_definition',
        'phase',
        'financing_type',
    ];

    /** @var array */
    protected $casts = [
        'financing_type' => 'array',   // JSON array: ["equity", "loan", ...]
        'sl_target'      => 'integer',
        'gl_target'      => 'integer',
        'phase'          => 'string',
    ];

    // =========================================================
    // SCOPES
    // =========================================================

    /** Bootcamp phase */
    public function scopeBootcamp($query)
    {
        return $query->where('phase', 'bootcamp');
    }

    /** Pitch phase */
    public function scopePitch($query)
    {
        return $query->where('phase', 'pitch');
    }

    /** Accelerator phase */
    public function scopeAccelerator($query)
    {
        return $query->where('phase', 'accelerator');
    }

    /** Capacity-building phase */
    public function scopeCapacityBuilding($query)
    {
        return $query->where('phase', 'capacity_building');
    }

    /** KPIs with SeedLab (SL) target */
    public function scopeHasSlTarget($query)
    {
        return $query->whereNotNull('sl_target');
    }

    /** KPIs with GrowthLab (GL) target */
    public function scopeHasGlTarget($query)
    {
        return $query->whereNotNull('gl_target');
    }

    /** Filter by financing type (e.g., 'loan') */
    public function scopeFinancingType($query, string $type)
    {
        return $query->whereJsonContains('financing_type', $type);
    }

    // =========================================================
    // ACCESSORS
    // =========================================================

    /** Human-readable phase */
    public function getPhaseLabelAttribute(): string
    {
        return match ($this->phase) {
            'bootcamp'           => 'Bootcamp',
            'pitch'              => 'Pitch',
            'accelerator'        => 'Accelerator',
            'capacity_building'  => 'Capacity Building',
            default              => ucfirst(str_replace('_', ' ', $this->phase ?? '')),
        };
    }

    /** Progress percentage for SeedLab (SL) */
    public function getSlProgressAttribute(): ?float
    {
        if (is_null($this->sl_target) || $this->sl_target == 0) {
            return null;
        }

        // Example: actual value from query (you'll compute in service/report)
        // Here we assume a helper method or pass via attribute
        // For now, return null – real value computed in KPI service
        return null;
    }

    /** Progress percentage for GrowthLab (GL) */
    public function getGlProgressAttribute(): ?float
    {
        if (is_null($this->gl_target) || $this->gl_target == 0) {
            return null;
        }
        return null;
    }

    /** Check if financing type includes a value */
    public function hasFinancingType(string $type): bool
    {
        return in_array($type, $this->financing_type ?? []);
    }
}