@php
    $layout = auth()->user()->role === 'super_admin' ? 'layouts.admin' : 'layouts.app';
@endphp

@extends($layout)

@section('title', 'Component 1 – Executive Dashboard')

@push('styles')
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <style>
        /* --- PowerBI / DHIS2 Professional Palette --- */
        :root {
            --bi-bg: #f3f4f6;
            --bi-card-bg: #ffffff;
            --bi-text-main: #1f2937;
            --bi-text-muted: #6b7280;
            --bi-border: #e5e7eb;
            
            /* DHIS2 Colors Preserved */
            --dhis2-blue: #2c6693;
            --dhis2-teal: #00796b;
            --dhis2-green: #4caf50;
            --dhis2-orange: #ff9800;
        }

        body {
            background-color: var(--bi-bg);
            font-family: 'Segoe UI', 'Roboto', 'Helvetica Neue', sans-serif;
            color: var(--bi-text-main);
        }

        /* --- Layout Utilities --- */
        .bi-container {
            padding: 1.5rem;
            max-width: 1920px; /* Allow wide screens */
            margin: 0 auto;
        }

        .gap-4 { gap: 1.5rem !important; }
        .mb-4 { margin-bottom: 1.5rem !important; }

        /* --- Header & Control Bar --- */
        .bi-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }

        .bi-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--dhis2-blue);
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .bi-controls {
            background: white;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            border: 1px solid var(--bi-border);
            display: flex;
            gap: 1rem;
            align-items: center;
            box-shadow: 0 1px 2px rgba(0,0,0,0.05);
        }

        .control-item {
            font-size: 0.85rem;
            color: var(--bi-text-muted);
            font-weight: 500;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        /* --- Professional Cards --- */
        .bi-card {
            background: var(--bi-card-bg);
            border-radius: 8px;
            border: 1px solid var(--bi-border);
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
            height: 100%;
            display: flex;
            flex-direction: column;
            overflow: hidden;
            transition: box-shadow 0.2s ease;
        }
        
        .bi-card:hover {
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }

        .bi-card-header {
            padding: 1rem 1.25rem;
            border-bottom: 1px solid var(--bi-border);
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #fcfcfd;
        }

        .bi-card-title {
            font-size: 0.95rem;
            font-weight: 600;
            color: var(--bi-text-main);
            margin: 0;
            text-transform: uppercase;
            letter-spacing: 0.025em;
        }

        .bi-card-body {
            padding: 1.25rem;
            flex: 1;
            position: relative;
        }

        /* --- KPI Cards (Top Row) --- */
        .kpi-card {
            position: relative;
            padding: 1.5rem;
            border-left: 4px solid transparent;
        }

        .kpi-card.blue { border-left-color: var(--dhis2-blue); }
        .kpi-card.teal { border-left-color: var(--dhis2-teal); }
        .kpi-card.green { border-left-color: var(--dhis2-green); }
        .kpi-card.orange { border-left-color: var(--dhis2-orange); }

        .kpi-label {
            font-size: 0.8rem;
            font-weight: 600;
            color: var(--bi-text-muted);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
        }

        .kpi-value {
            font-size: 2.25rem;
            font-weight: 700;
            color: var(--bi-text-main);
            line-height: 1;
        }

        .kpi-icon {
            position: absolute;
            right: 1.5rem;
            bottom: 1.5rem;
            font-size: 3rem;
            opacity: 0.1;
            color: var(--bi-text-main);
        }

        /* --- Data Tables (Professional Look) --- */
        .table-bi {
            width: 100%;
            margin-bottom: 0;
            font-size: 0.9rem;
        }

        .table-bi th {
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.75rem;
            letter-spacing: 0.05em;
            color: var(--bi-text-muted);
            background-color: #f9fafb;
            padding: 0.75rem 1rem;
            border-bottom: 1px solid var(--bi-border);
            border-top: none;
        }

        .table-bi td {
            padding: 0.75rem 1rem;
            vertical-align: middle;
            color: var(--bi-text-main);
            border-bottom: 1px solid #f3f4f6;
        }

        .table-bi tr:last-child td { border-bottom: none; }
        
        .fw-medium { font-weight: 500; }
        .text-right { text-align: right; }
        
        /* Progress bars inside tables */
        .tbl-progress {
            height: 6px;
            border-radius: 3px;
            background-color: #e5e7eb;
            width: 100px;
            display: inline-block;
            vertical-align: middle;
            margin-left: 10px;
            overflow: hidden;
        }
        .tbl-progress-bar {
            height: 100%;
            border-radius: 3px;
        }

        /* --- Tabs Navigation --- */
        .nav-tabs-bi {
            border-bottom: 1px solid var(--bi-border);
            padding: 0 1rem;
            background: #f9fafb;
            gap: 1rem;
            display: flex;
        }

        .nav-tabs-bi .nav-link {
            border: none;
            border-bottom: 2px solid transparent;
            color: var(--bi-text-muted);
            font-weight: 500;
            padding: 1rem 0.5rem;
            font-size: 0.9rem;
            background: transparent;
            margin-bottom: -1px;
            border-radius: 0;
            transition: all 0.2s;
        }

        .nav-tabs-bi .nav-link:hover {
            color: var(--dhis2-blue);
        }

        .nav-tabs-bi .nav-link.active {
            color: var(--dhis2-blue);
            border-bottom-color: var(--dhis2-blue);
            font-weight: 600;
            background: transparent;
        }

        /* Map Container */
        .map-wrapper {
            height: 350px;
            width: 100%;
            border-radius: 6px;
            overflow: hidden;
        }

        /* Buttons */
        .btn-action {
            padding: 0.4rem 0.8rem;
            font-size: 0.85rem;
            border-radius: 4px;
            transition: all 0.2s;
            text-decoration: none;
        }
        .btn-primary-bi {
            background-color: var(--dhis2-blue);
            color: white;
            border: 1px solid var(--dhis2-blue);
        }
        .btn-primary-bi:hover { background-color: #235276; color: white; }
        
        .btn-outline-bi {
            background-color: white;
            color: var(--dhis2-blue);
            border: 1px solid var(--dhis2-blue);
        }
        .btn-outline-bi:hover { background-color: #f0f7ff; }

    </style>
@endpush

@section('content')
    <div class="bi-container">
        
        <div class="bi-header">
            <div>
                <h1 class="bi-title">
                    <i class="fas fa-chart-line"></i> Component 1 Dashboard
                </h1>
                <nav aria-label="breadcrumb" style="margin-top: 5px;">
                    <ol class="breadcrumb mb-0" style="font-size: 0.85rem;">
                        <li class="breadcrumb-item"><a href="{{ route('home') }}" class="text-muted text-decoration-none">Home</a></li>
                        <li class="breadcrumb-item active text-primary">Overview</li>
                    </ol>
                </nav>
            </div>
            
            <div class="d-flex gap-2">
                <div class="bi-controls d-none d-md-flex">
                    <div class="control-item">
                        <i class="far fa-calendar-alt"></i> Dec 2025
                    </div>
                    <div style="width: 1px; height: 15px; background: #ddd;"></div>
                    <div class="control-item">
                        <i class="fas fa-filter"></i> All Regions
                    </div>
                </div>

                <a href="{{ route('component1.groups.index') }}" class="btn btn-outline-bi btn-action">
                    <i class="fas fa-users me-1"></i> Manage Groups
                </a>
                <a href="{{ route('component1.members.index') }}" class="btn btn-primary-bi btn-action">
                    <i class="fas fa-user-friends me-1"></i> Members
                </a>
                <a href="{{ route('component1.vsla.report') }}" class="btn btn-primary-bi btn-action">
                    <i class="fas fa-file-excel me-1"></i> Monthly Report
                </a>
            </div>
        </div>

        <div class="row g-4 mb-4">
            <div class="col-xl-3 col-md-6">
                <div class="bi-card kpi-card green">
                    <div class="kpi-label">Active Groups</div>
                    <div class="kpi-value">{{ number_format($stats['total_groups']) }}</div>
                    <div class="text-success mt-2 small"><i class="fas fa-arrow-up"></i> 96.25% of Target</div>
                    <i class="fas fa-users kpi-icon"></i>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="bi-card kpi-card blue">
                    <div class="kpi-label">Total Members</div>
                    <div class="kpi-value">{{ number_format($stats['total_members']) }}</div>
                    <div class="text-muted mt-2 small">across all counties</div>
                    <i class="fas fa-id-card kpi-icon"></i>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="bi-card kpi-card teal">
                    <div class="kpi-label">Members Trained</div>
                    <div class="kpi-value">{{ number_format($trainingCompletion['Module 1'] ?? 0) }}</div>
                    <div class="text-info mt-2 small">Completed Module 1</div>
                    <i class="fas fa-graduation-cap kpi-icon"></i>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="bi-card kpi-card orange">
                    <div class="kpi-label">Total Supports</div>
                    <div class="kpi-value">{{ number_format($kpiStats['supports']) }}</div>
                    <div class="text-warning mt-2 small">Kits & Assets distributed</div>
                    <i class="fas fa-hands-helping kpi-icon"></i>
                </div>
            </div>
        </div>

        <div class="row g-4">
            
            <div class="col-lg-8">
                <div class="bi-card mb-4">
                    <div class="bi-card-header">
                        <h6 class="bi-card-title">VSLA Engagement Performance</h6>
                        <div class="text-muted small"><i class="fas fa-info-circle"></i> Target vs Achieved</div>
                    </div>
                    <div class="bi-card-body">
                        <div class="row align-items-center">
                            <div class="col-md-7">
                                <table class="table table-bi table-hover">
                                    <thead>
                                        <tr>
                                            <th>Category</th>
                                            <th class="text-center">Target</th>
                                            <th class="text-center">Achieved</th>
                                            <th class="text-right">Progress</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach(['matured', 'progressive', 'new'] as $nature)
                                        @php 
                                            $target = $nature === 'matured' ? 240 : ($nature === 'progressive' ? 160 : 400);
                                            $val = $groupsByNature->where('nature', $nature)->first()->achieved ?? 0;
                                            $pct = $target > 0 ? ($val / $target) * 100 : 0;
                                            $color = $pct >= 100 ? '#4caf50' : '#2c6693';
                                        @endphp
                                        <tr>
                                            <td class="text-capitalize fw-medium">{{ $nature }}</td>
                                            <td class="text-center text-muted">{{ $target }}</td>
                                            <td class="text-center fw-bold">{{ $val }}</td>
                                            <td class="text-right">
                                                <span class="small fw-bold me-1">{{ round($pct) }}%</span>
                                                <div class="tbl-progress">
                                                    <div class="tbl-progress-bar" style="width: {{ $pct }}%; background-color: {{ $color }};"></div>
                                                </div>
                                            </td>
                                        </tr>
                                        @endforeach
                                        <tr class="bg-light">
                                            <td class="fw-bold">Total</td>
                                            <td class="text-center fw-bold">800</td>
                                            <td class="text-center fw-bold">{{ $groupsByNature->sum('achieved') }}</td>
                                            <td class="text-right fw-bold text-success">{{ round(($groupsByNature->sum('achieved') / 800) * 100, 1) }}%</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                            <div class="col-md-5">
                                <div style="height: 200px;">
                                    <canvas id="vslaCategoryChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="bi-card">
                    <div class="bi-card-header p-0">
                        <ul class="nav nav-tabs-bi" id="dataTabs" role="tablist">
                            <li class="nav-item">
                                <a class="nav-link active" id="tab-membership" data-bs-toggle="tab" href="#content-membership" role="tab">Membership</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="tab-trainings" data-bs-toggle="tab" href="#content-trainings" role="tab">Trainings</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="tab-kits" data-bs-toggle="tab" href="#content-kits" role="tab">Kits</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" id="tab-assets" data-bs-toggle="tab" href="#content-assets" role="tab">Assets & Business</a>
                            </li>
                        </ul>
                    </div>
                    <div class="bi-card-body">
                        <div class="tab-content" id="dataTabsContent">
                            
                            <div class="tab-pane fade show active" id="content-membership" role="tabpanel">
                                <div class="row">
                                    <div class="col-md-8">
                                        <table class="table table-bi table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Category</th>
                                                    <th class="text-right">Female</th>
                                                    <th class="text-right">Male</th>
                                                    <th class="text-right">Adol. Girls</th>
                                                    <th class="text-right">Total</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                @foreach(['matured', 'progressive', 'new'] as $nature)
                                                <tr>
                                                    <td class="text-capitalize">{{ $nature }}</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2[$nature]['female'] ?? 0) }}</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2[$nature]['male'] ?? 0) }}</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2[$nature]['adol_girls'] ?? 0) }}</td>
                                                    <td class="text-right fw-bold">{{ number_format($membershipByNature2[$nature]['total'] ?? 0) }}</td>
                                                </tr>
                                                @endforeach
                                                <tr class="bg-light fw-bold">
                                                    <td>Total</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2->sum('female')) }}</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2->sum('male')) }}</td>
                                                    <td class="text-right">{{ number_format($membershipByNature2->sum('adol_girls')) }}</td>
                                                    <td class="text-right text-primary">{{ number_format($membershipByNature2->sum('total')) }}</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                    </div>
                                    <div class="col-md-4 d-flex align-items-center justify-content-center">
                                        <div style="height: 180px; width: 100%;">
                                            <canvas id="membershipDoughnut"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="content-trainings" role="tabpanel">
                                <div class="row">
                                    <div class="col-md-12">
                                        <h6 class="mb-3 text-muted text-uppercase small ls-1">Cumulative Members Trained</h6>
                                        <table class="table table-bi table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Category</th>
                                                    <th class="text-right">Female</th>
                                                    <th class="text-right">Male</th>
                                                    <th class="text-right">Adol. Girls</th>
                                                    <th class="text-right">Total</th>
                                                    <th class="text-right" style="width: 150px;">Target Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                @foreach(['matured', 'progressive', 'new'] as $nature)
                                                <tr>
                                                    <td class="text-capitalize">{{ $nature }}</td>
                                                    <td class="text-right">{{ number_format($trainingsCumulative[$nature]['female'] ?? 0) }}</td>
                                                    <td class="text-right">{{ number_format($trainingsCumulative[$nature]['male'] ?? 0) }}</td>
                                                    <td class="text-right">{{ number_format($trainingsCumulative[$nature]['adol_girls'] ?? 0) }}</td>
                                                    <td class="text-right fw-bold">{{ number_format($trainingsCumulative[$nature]['total'] ?? 0) }}</td>
                                                    <td class="text-right">
                                                        <span class="badge bg-success bg-opacity-10 text-success">On Track</span>
                                                    </td>
                                                </tr>
                                                @endforeach
                                            </tbody>
                                        </table>
                                        <div style="height: 250px; margin-top: 1rem;">
                                            <canvas id="trainingsCumulativeChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="content-kits" role="tabpanel">
                                <div class="row g-4">
                                    <div class="col-md-6 border-end">
                                        <h6 class="text-center text-muted small text-uppercase mb-3">Groups Received Kits</h6>
                                        <div style="height: 200px;">
                                            <canvas id="kitsGroupsChart"></canvas>
                                        </div>
                                        <div class="text-center mt-3">
                                            <h3 class="fw-bold mb-0">{{ $kitsGroupsCumulative->sum('groups') }}</h3>
                                            <small class="text-muted">Total Groups</small>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <h6 class="text-center text-muted small text-uppercase mb-3">Members Benefited</h6>
                                        <div style="height: 200px;">
                                            <canvas id="kitsMembersChart"></canvas>
                                        </div>
                                        <div class="text-center mt-3">
                                            <h3 class="fw-bold mb-0">{{ $kitsMembersCumulative->sum('total_female') + $kitsMembersCumulative->sum('total_male') }}</h3>
                                            <small class="text-muted">Total Individuals</small>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="tab-pane fade" id="content-assets" role="tabpanel">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="p-3 border rounded bg-light mb-3">
                                            <h6 class="text-dark mb-1">Productive Assets</h6>
                                            <p class="text-muted small mb-0">Groups that received productive assets</p>
                                        </div>
                                        <table class="table table-bi table-sm">
                                            <thead>
                                                <tr>
                                                    <th>Category</th>
                                                    <th class="text-right">Groups</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                @foreach(['matured', 'progressive', 'new'] as $nature)
                                                <tr>
                                                    <td class="text-capitalize">{{ $nature }}</td>
                                                    <td class="text-right fw-bold">{{ $productiveAssetGroups[$nature]['groups'] ?? 0 }}</td>
                                                </tr>
                                                @endforeach
                                                <tr class="fw-bold bg-light">
                                                    <td>Total</td>
                                                    <td class="text-right">{{ $productiveAssetGroups->sum('groups') }}</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="p-3 border rounded bg-light mb-3">
                                            <h6 class="text-dark mb-1">New Businesses</h6>
                                            <p class="text-muted small mb-0">Groups operating new businesses</p>
                                        </div>
                                        <div style="height: 200px;">
                                            <canvas id="newBusinessChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>

                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4">
                
                <div class="bi-card mb-4">
                    <div class="bi-card-header">
                        <h6 class="bi-card-title">Geographic Distribution</h6>
                        <button class="btn btn-sm btn-link text-muted p-0"><i class="fas fa-expand"></i></button>
                    </div>
                    <div class="bi-card-body p-0">
                        <ul class="nav nav-pills nav-fill p-2 bg-light small" id="mapTabs" role="tablist">
                            <li class="nav-item" role="presentation">
                                <button class="nav-link active py-1" id="group-map-tab" data-bs-toggle="pill" data-bs-target="#group-map-pane" type="button">Groups</button>
                            </li>
                            <li class="nav-item" role="presentation">
                                <button class="nav-link py-1" id="member-map-tab" data-bs-toggle="pill" data-bs-target="#member-map-pane" type="button">Members</button>
                            </li>
                        </ul>
                        <div class="tab-content">
                            <div class="tab-pane fade show active" id="group-map-pane">
                                <div id="groupMap" style="height: 300px;"></div>
                            </div>
                            <div class="tab-pane fade" id="member-map-pane">
                                <div id="memberMap" style="height: 300px;"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="bi-card mb-4">
                    <div class="bi-card-header">
                        <h6 class="bi-card-title">Gender Balance</h6>
                    </div>
                    <div class="bi-card-body">
                        <div style="height: 220px;">
                            <canvas id="genderChart"></canvas>
                        </div>
                        <div class="d-flex justify-content-center gap-4 mt-3">
                            <div class="text-center">
                                <span class="d-block text-muted small">Female</span>
                                <span class="fw-bold text-dark">{{ number_format($genderData['female']) }}</span>
                            </div>
                            <div class="text-center">
                                <span class="d-block text-muted small">Male</span>
                                <span class="fw-bold text-dark">{{ number_format($genderData['male']) }}</span>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="bi-card">
                    <div class="bi-card-header">
                        <h6 class="bi-card-title">Top Counties</h6>
                    </div>
                    <div class="bi-card-body">
                        <div style="height: 250px;">
                            <canvas id="groupsByCountyChart"></canvas>
                        </div>
                    </div>
                </div>

            </div>
        </div>

        <div class="d-none">
             <canvas id="trainingChart"></canvas> <canvas id="productiveAssetChart"></canvas>
        </div>
    </div>
@endsection

@push('scripts')
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <script>
        // --- Professional Configuration ---
        const theme = {
            colors: {
                primary: '#2c6693',
                secondary: '#00796b',
                success: '#4caf50',
                warning: '#ff9800',
                danger: '#f44336',
                info: '#00bcd4',
                slate: '#64748b',
                light: '#f1f5f9'
            },
            fonts: {
                main: "'Segoe UI', 'Roboto', sans-serif"
            }
        };

        Chart.defaults.font.family = theme.fonts.main;
        Chart.defaults.font.size = 11;
        Chart.defaults.color = '#64748b';
        Chart.defaults.plugins.legend.labels.usePointStyle = true;
        Chart.defaults.plugins.legend.labels.boxWidth = 8;
        Chart.defaults.scale.grid.color = '#f1f5f9';

        // --- Charts Initialization ---

        // 1. VSLA Category Bar Chart (Horizontal for better label readability)
        new Chart(document.getElementById('vslaCategoryChart'), {
            type: 'bar',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [{
                    label: 'Achieved',
                    data: [
                        {{ $groupsByNature->where('nature', 'matured')->first()->achieved ?? 0 }},
                        {{ $groupsByNature->where('nature', 'progressive')->first()->achieved ?? 0 }},
                        {{ $groupsByNature->where('nature', 'new')->first()->achieved ?? 0 }}
                    ],
                    backgroundColor: theme.colors.primary,
                    borderRadius: 4,
                    barPercentage: 0.6
                }]
            },
            options: {
                indexAxis: 'y', // Horizontal bars look more professional for comparison
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: {
                    x: { grid: { display: true } },
                    y: { grid: { display: false } }
                }
            }
        });

        // 2. Membership Doughnut
        new Chart(document.getElementById('membershipDoughnut'), {
            type: 'doughnut',
            data: {
                labels: ['Female', 'Male', 'Adolescent'],
                datasets: [{
                    data: [{{ $totalFemale ?? 17986 }}, {{ $totalMale ?? 2245 }}, {{ $totalAdolGirls ?? 3304 }}],
                    backgroundColor: [theme.colors.success, theme.colors.warning, theme.colors.info],
                    borderWidth: 0,
                    hoverOffset: 5
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '70%',
                plugins: {
                    legend: { position: 'bottom', labels: { padding: 20 } }
                }
            }
        });

        // 3. Gender Chart (Doughnut)
        new Chart(document.getElementById('genderChart'), {
            type: 'doughnut',
            data: {
                labels: ['Female', 'Male'],
                datasets: [{
                    data: [{{ $genderData['female'] }}, {{ $genderData['male'] }}],
                    backgroundColor: [theme.colors.info, theme.colors.primary],
                    borderWidth: 0,
                    cutout: '75%'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } }
            }
        });

        // 4. Trainings Cumulative
        new Chart(document.getElementById('trainingsCumulativeChart'), {
            type: 'bar',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [
                    { label: 'Female', data: [{{ $trainingsCumulative['matured']['female'] ?? 0 }}, {{ $trainingsCumulative['progressive']['female'] ?? 0 }}, {{ $trainingsCumulative['new']['female'] ?? 0 }}], backgroundColor: theme.colors.success },
                    { label: 'Male', data: [{{ $trainingsCumulative['matured']['male'] ?? 0 }}, {{ $trainingsCumulative['progressive']['male'] ?? 0 }}, {{ $trainingsCumulative['new']['male'] ?? 0 }}], backgroundColor: theme.colors.warning }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: { x: { stacked: true }, y: { stacked: true } }
            }
        });

        // 5. Kits Groups
        new Chart(document.getElementById('kitsGroupsChart'), {
            type: 'bar',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [{
                    label: 'Groups',
                    data: [{{ $kitsGroupsCumulative['matured']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['progressive']['groups'] ?? 0 }}, {{ $kitsGroupsCumulative['new']['groups'] ?? 0 }}],
                    backgroundColor: theme.colors.secondary,
                    borderRadius: 4
                }]
            },
            options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
        });

        // 6. Kits Members
        new Chart(document.getElementById('kitsMembersChart'), {
            type: 'line',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [{
                    label: 'Members',
                    data: [{{ $kitsMembersCumulative['matured']['total'] ?? 0 }}, {{ $kitsMembersCumulative['progressive']['total'] ?? 0 }}, {{ $kitsMembersCumulative['new']['total'] ?? 0 }}],
                    borderColor: theme.colors.primary,
                    backgroundColor: 'rgba(44, 102, 147, 0.1)',
                    fill: true,
                    tension: 0.4
                }]
            },
            options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
        });

        // 7. New Business
        new Chart(document.getElementById('newBusinessChart'), {
            type: 'bar',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [{
                    label: 'Businesses',
                    data: [{{ $newBusinessGroups['matured']['groups'] ?? 0 }}, {{ $newBusinessGroups['progressive']['groups'] ?? 0 }}, {{ $newBusinessGroups['new']['groups'] ?? 0 }}],
                    backgroundColor: theme.colors.success,
                    borderRadius: 4
                }]
            },
            options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { display: false } } }
        });

        // 8. Groups by County (Top 10)
        const groupsByCounty = {!! json_encode($groupsByCounty->pluck('county')) !!};
        const groupsByCountyTotal = {!! json_encode($groupsByCounty->pluck('total')) !!};
        new Chart(document.getElementById('groupsByCountyChart'), {
            type: 'bar',
            data: {
                labels: groupsByCounty,
                datasets: [{
                    label: 'Groups',
                    data: groupsByCountyTotal,
                    backgroundColor: theme.colors.primary,
                    borderRadius: 3
                }]
            },
            options: {
                indexAxis: 'y',
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: false } },
                scales: { x: { grid: { display: true } }, y: { grid: { display: false } } }
            }
        });

        // 9. Productive Asset (Hidden but initialized)
        new Chart(document.getElementById('productiveAssetChart'), {
            type: 'bar',
            data: {
                labels: ['Mature', 'Progressive', 'New'],
                datasets: [{ label: 'Assets', data: [{{ $productiveAssetGroups['matured']['groups'] ?? 0 }}, {{ $productiveAssetGroups['progressive']['groups'] ?? 0 }}, {{ $productiveAssetGroups['new']['groups'] ?? 0 }}], backgroundColor: theme.colors.primary }]
            }
        });
        
        // 10. Training (Hidden but initialized)
        new Chart(document.getElementById('trainingChart'), {
            type: 'bar',
            data: {
                labels: {!! json_encode(array_keys($trainingCompletion)) !!},
                datasets: [{ label: 'Completed', data: {!! json_encode(array_values($trainingCompletion)) !!}, backgroundColor: theme.colors.warning }]
            }
        });

        // --- MAPS ---
        const groupMap = L.map('groupMap', { zoomControl: false }).setView([7.0, 30.0], 6);
        L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png').addTo(groupMap);
        const groupLocations = {!! json_encode($groupLocations) !!};
        groupLocations.forEach(loc => {
            L.circleMarker([loc.lat, loc.lon], { radius: 5, fillColor: theme.colors.primary, color: '#fff', weight: 1, opacity: 1, fillOpacity: 0.8 }).addTo(groupMap).bindPopup(loc.name);
        });

        const memberMap = L.map('memberMap', { zoomControl: false }).setView([7.0, 30.0], 6);
        L.tileLayer('https://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}{r}.png').addTo(memberMap);
        const memberLocations = {!! json_encode($memberLocations) !!};
        memberLocations.forEach(loc => {
            L.circleMarker([loc.lat, loc.lon], { radius: 3, fillColor: theme.colors.success, color: '#fff', weight: 0.5, opacity: 1, fillOpacity: 0.6 }).addTo(memberMap).bindPopup(loc.name);
        });

        // --- Utility: Fix Chart.js resizing in Tabs ---
        // When a tab is clicked, trigger a resize to ensure charts render correctly in the new visible area
        const tabEls = document.querySelectorAll('button[data-bs-toggle="tab"], button[data-bs-toggle="pill"]');
        tabEls.forEach(tab => {
            tab.addEventListener('shown.bs.tab', function (event) {
                window.dispatchEvent(new Event('resize'));
                // Specific map invalidation
                if(event.target.id === 'group-map-tab') groupMap.invalidateSize();
                if(event.target.id === 'member-map-tab') memberMap.invalidateSize();
            });
        });

    </script>
@endpush